/*
check-user.c
valid user-checking program, to be used with linuxmagic.com open source magic-smtpd

or with Jay Soffian's $RCPTCHECK patch to the stock qmail-smtpd.c
by Gregory nowak
version 0.2c, 28 May, 2008



command line arguments: accepts 1 command line argument in the form user@domain or checks $RECIPIENT if qmail is defined

return values: 0 if user exists, 
100 if qmail is defined, and user doesn't exist,
or 1 if magic is defined, and user doesn't exist


Copyright (C) 2004-2008 Gregory Nowak <http://www.romuald.net.eu.org/>.

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.
You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

Gregory Nowak <gnowak@users.sourceforge.net>

*/


/*
changes from 0.2b to 0.2c

check if user@domain is defined in virtualdomains, and assume that address to be valid if it is found
we don't check for a domain preappend anymore in virtualdomains,
the domain itself, or an e-mail address are considered as valid if they're found

changes from 0.2a to 0.2b

check if $RELAYCLIENT is defined,
before refusing to accept the message

 changes from 0.2 to 0.2a

fixed a bug where user-something@defaultdomain was returned as valid,
even if that address didn't exist

fixed a bug when checking for user-something-default@defaultdomain

changes from 0.1 to 0.2

modified program to be used
with the $RCPTCHECK patch to qmail-smtpd, while retaining functionality with magic-smtpd from version 0.1

fixed a bug where if a user was valid for the local domain,
that user would also be returned as valid for virtual domains

*/



/* user-editable values */
#define max_user_name 81	// max length of user name
#define max_domain_name 81	// max length of domain/domain name
#define qmail 100            // invalid user return value for qmail-smtpd
// #define magic 1			// invalid user return value for magic-smtpd



const char alias_dir[] = "/var/qmail/alias\0";	// alias directory
const char v_domains_file[] = "/var/qmail/control/virtualdomains\0";	// qmail's virtualdomains file
const char locals_file[] = "/var/qmail/control/locals\0";	// qmail's locals file
/* user-editable section ends here */



#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <pwd.h>
#include <sys/types.h>
#include <unistd.h>

#ifdef qmail
#include <stdlib.h>
#endif


const char null = '\0';		// different from NULL


// prototypes
int check_user_name (const char *email);	// check against /etc/passwd
int check_user_alias (const char *home_dir, const char *ext);	// check .qmail alias files in user's home directory and in alias_dir
int check_alias (const char *user, const char *domain);	// specific check against alias_dir
int check_vdomain (const char *domain);	// check against v_domains_file
int check_locals (const char *domain);	// check against locals_file
char *strlwr (char *s);		// convert string from upper to lower case


int
main (int argc, char *argv[])
{


#ifdef qmail
  int status = 100;
#endif

#ifdef magic
  int status = 1;		// return value
#endif


  // if $RELAYCLIENT is set, we stop here
  if (getenv ("RELAYCLIENT\0") != NULL)
    {
      status = 0;
      return status;
    }


  if (argc == 1)
    {
#ifdef qmail
      argv[1] = getenv ("RECIPIENT\0");
      if (argv[1] == NULL)
	return status;
#endif


    }
  if (argv[1] == NULL)
    return status;

  if (strlen (argv[1]) > max_user_name + max_domain_name)
    return status;		// rcpt to is too long


  strlwr (argv[1]);		// convert to lower case


  status = check_user_name (argv[1]);

  return status;
}





int
check_user_name (const char *email)
{				// check user name against /etc/passwd, calling other functions as needed
// email should be in the form user@domain
// returns 0 on success, 1 or 100 otherwise

  char domain[max_domain_name];	// holds domain name
  char name[max_user_name];	// holds user name
  char before_dash[max_user_name];	// holds first part of user name with a dash
  char after_dash[max_user_name];	// holds second part of user name with a dash
  char temp[max_user_name];
  const char dash[] = "-\0";
  const char at_sign[] = "@\0";

#ifdef qmail
  int status = 100;
#endif
#ifdef magic
  int status = 1;		// return value
#endif

  struct passwd *u_name;	// struct passed to getpwnam function



  domain[0] = null;
  name[0] = null;
  before_dash[0] = null;
  after_dash[0] = null;
  temp[0] = null;


// find and store the domain name from the command line argument, throwing away the "@"
  if (strstr (email, at_sign) == NULL)
    return status;		//we won't check /var/qmail/control/defaultdomain, or /etc/magic-mail/control/defaultdomain
  if (strlen (strstr (email, at_sign) + 1) >= max_domain_name)
    return status;		// domain name is too long
  strcpy (domain, strstr (email, at_sign) + 1);
  if (domain == NULL)
    return status;

// copy the user name to name, based on the lengths of email and of domain
  if ((strlen (email) - strlen (domain) - 1) > max_user_name)
    return status;
  strncpy (name, email, (strlen (email) - strlen (domain) - 1));
  name[(strlen (email) - strlen (domain) - 1)] = null;
  if (name == NULL)
    return status;


// if our user name has a dash, this is the time to find that out
  if (strstr (name, dash) != NULL)
    {				// we had a dash in name
      if (strlen (strstr (name, dash)) > max_user_name)
	return status;
      strcpy (after_dash, strstr (name, dash) + 1);
      if ((strlen (name) - strlen (after_dash)) - 1 > max_user_name)
	return status;
      strncpy (before_dash, name, (strlen (name) - strlen (after_dash)) - 1);
      before_dash[(strlen (name) - strlen (after_dash)) - 1] = '\0';


    }				// if ( strstr( name, dash) != NULL)

  if (before_dash[0] == null)
    u_name = getpwnam (name);
  else
    u_name = getpwnam (before_dash);

// find out if name exists
  if (u_name == NULL)
    {				// check for aliases
      if (after_dash[0] != '\0')
	if ((status = check_alias (name, domain)) == 0)
	  return status;


      if (before_dash[0] == null)
	if ((status = check_alias (name, domain)) == 0)
	  return status;
	else if ((status = check_alias (before_dash, domain)) == 0)
	  return status;
	else
	  status = check_vdomain (domain);
// if check_vdomain returned status other than 0,
// we pass it the full email address before giving up
      if (status != 0)
	status = check_vdomain (email);
      return status;
    }				// if (u_name == NULL)


  if (before_dash[0] == null)
    {
      if ((strcmp (name, u_name->pw_name) == 0) && (after_dash[0] == null))
	{			// we got an exact match with no extension
	  status = 0;
	  status = check_locals (domain);
	  return status;
	}
    }

  else if ((strcmp (before_dash, u_name->pw_name) == 0)
	   && (after_dash[0] == null))
    {				// we got an exact match with no extension
      status = 0;
      status = check_locals (domain);
      return status;
    }





  if ((strcmp (before_dash, u_name->pw_name) == 0) && (after_dash[0] != null))
    {				// we got an exact match with an extension
      if (((status = check_user_alias (u_name->pw_dir, after_dash)) == 0)
	  && ((status = check_locals (domain)) == 0))
	return status;


// check for a after_dash-default file
      strcat (after_dash, "-default\0");
      if (((status = check_user_alias (u_name->pw_dir, after_dash)) == 0)
	  && ((status = check_locals (domain)) == 0))
	return status;


// check for a .qmail-default file
      strcpy (after_dash, "default\0");
      if (((status = check_user_alias (u_name->pw_dir, after_dash)) == 0)
	  && ((status = check_locals (domain)) == 0))
	return status;

// check for a default file, without the last part after the dash
// for example, list-accept-something@example.com would be as .qmail-accept-default,
// and not as .qmail-accept-something-default in list's home directory


      strcpy (after_dash, strrchr (name, '-') + 1);	// get the last part after the dash, which we don't really want
      strncpy (temp, name, (strlen (name) - strlen (after_dash)));
      temp[(strlen (name) - strlen (after_dash))] = null;
      strcat (temp, "default\0");

      status = check_user_alias (u_name->pw_dir, temp);

    }



  if (status != 0)
    return status;

  status = check_locals (domain);
  return status;
}





int
check_user_alias (const char *home_dir, const char *ext)
{				// check for the existence of home_dir/.qmail-ext file
// return 0 on success, 1 or 100 otherwise

  char file_name[max_user_name + max_domain_name];	// path to .qmail file, this should be more then big enough
#ifdef qmail
  int status = 100;
#endif
#ifdef magic
  int status = 1;		// return value
#endif

  file_name[0] = null;


// concatenate the path to the .qmail file in file_name

  if ((strlen (file_name) + strlen (home_dir)) >
      max_user_name + max_domain_name)
    return status;
// this check should never be true

  strcat (file_name, home_dir);

  if ((strlen (file_name) + strlen ("/.qmail-\0")) >
      max_user_name + max_domain_name)
    return status;
  strcat (file_name, "/.qmail-\0");

  if ((strlen (file_name) + strlen (ext)) > max_user_name + max_domain_name)
    return status;
  strcat (file_name, ext);

// ok, we should now have a full path to a .qmail file, let's see if it exists

  status = access (file_name, 0);
  if (status == -1)
    {
#ifdef qmail
      status = 100;
#endif
#ifdef magic
      status = 1;
#endif
    }

  return status;

}





int
check_alias (const char *user, const char *domain)
{				// checks if user exists in alias_dir
// user should contain the user name.
// domain should contain a FQDN

// returns 0 on success, 1 or 100 otherwise


  char domain_name[max_domain_name];	// for copying domain into
  char temp[max_user_name + max_domain_name];
  char after_dash[max_user_name];	// holds second part of user name with a dash
  const char period[] = ".\0";
#ifdef qmail
  int status = 100;
#endif
#ifdef magic
  int status = 1;		// return value
#endif
  int size = 0;

  domain_name[0] = null;
  temp[0] = null;
  after_dash[0] = null;


// let's edit domain first, to check for virtual domain users

  if (strstr (domain, period) == NULL)
    return status;		// shouldn't happen
  if (strlen (strstr (domain, period)) > max_domain_name)
    return status;		// this should never happen
  strcpy (domain_name, strstr (domain, period));

// we've got everything including and after the period in domain_name, however, since this isn't what we want ...
  size = strlen (domain_name);

  strncpy (domain_name, domain, (strlen (domain) - size));

  domain_name[strlen (domain) - size] = '\0';

  if (domain_name == NULL)
    return status;		// this shouldn't happen

// ok, the first part of the fqdn should be in domain_name

// we check virtual domain users
// of the form alias_dir/.qmail-domain_name-user, or alias_dir/.qmail-domain_name-user-default

  strcpy (temp, domain_name);
  strcat (temp, "-\0");
  strcat (temp, user);

  if ((status = check_user_alias (alias_dir, temp)) == 0)
    return status;



  if (strstr (user, "-\0") != NULL)
    {
// check for a default file, without the last part after the dash
// for example, list-accept-something@example.com would be as .qmail-accept-default,
// and not as .qmail-accept-something-default in list's home directory


      strcpy (temp, domain_name);
      strcat (temp, "-\0");
      strcpy (after_dash, strrchr (user, '-') + 1);	// get the last part after the dash, which we don't really want
      strncat (temp, user, (strlen (user) - strlen (after_dash)));
      temp[(strlen (domain_name) + 1) +
	   (strlen (user) - strlen (after_dash))] = null;
      strcat (temp, "default\0");
      if ((status = check_user_alias (alias_dir, temp)) == 0)
	return status;
    }



  strcpy (temp, domain_name);
  strcat (temp, "-default\0");

  if ((status = check_user_alias (alias_dir, temp)) == 0)
    return status;




// if domain doesn't exist in locals_file,
// there's no point in continuing
  if ((status = check_locals (domain)) != 0)
    return status;


// now, let's see if alias_dir/.qmail-user exists

  if ((status = check_user_alias (alias_dir, user)) == 0)
    return status;



  if (strstr (user, "-\0") != NULL)
    {
// check for a default file, without the last part after the dash
// for example, list-accept-something@example.com would be as .qmail-accept-default,
// and not as .qmail-accept-something-default in list's home directory


      strcpy (after_dash, strrchr (user, '-'));	// get the last part after the dash, which we don't really want
      strncpy (temp, user, (strlen (user) - strlen (after_dash)));
      temp[(strlen (user) - strlen (after_dash))] = null;
      strcat (temp, "-default\0");
      if ((status = check_user_alias (alias_dir, temp)) == 0)
	return status;
    }



// check if alias_dir/.qmail-user-default exists
  strcpy (temp, user);
  strcat (temp, "-default\0");

  if ((status = check_user_alias (alias_dir, temp)) == 0)
    return status;



// see if alias_dir/.qmail-default exists
  strcpy (temp, "default\0");
  if ((status = check_user_alias (alias_dir, temp)) == 0)
    return status;



// ok, we're done

  return status;
}





int
check_vdomain (const char *domain)
{				// checks if domain exists in v_domains_file
// if it does, then we return 0, user exists
// if not, we return 1 or 100
// domain has to be a fqdn, or a full e-mail address


  char input[max_user_name + max_domain_name];
  FILE *f_stream;


  if (domain == NULL)
    {
#ifdef qmail
      return 100;
#endif
#ifdef magic
      return 1;
#endif
    }

  if ((f_stream = fopen (v_domains_file, "r")) == NULL)
    {
#ifdef qmail
      return 100;
#endif
#ifdef magic
      return 1;
#endif
    }


  for (; feof (f_stream) == 0;
       fgets (input, max_user_name + max_domain_name, f_stream))
    if (strncmp (domain, input, strlen (domain)) == 0)
      {				// got a match
	fclose (f_stream);
	return 0;
      }

  fclose (f_stream);
#ifdef qmail
  return 100;
#endif
  return 1;
}





int
check_locals (const char *domain)
{				// checks if domain exists in locals_file
// if it does, then we return 0, user exists
// if not, we return 1 or 100
// domain has to be a fqdn



  char input[max_user_name + max_domain_name];
  FILE *f_stream;

  if ((f_stream = fopen (locals_file, "r")) == NULL)
    {
#ifdef qmail
      return 100;
#endif
#ifdef magic
      return 1;
#endif
    }


  for (; feof (f_stream) == 0;
       fgets (input, max_user_name + max_domain_name, f_stream))
    if (strncmp (domain, input, strlen (domain)) == 0)
      {				// got a match
	fclose (f_stream);
	return 0;
      }

  fclose (f_stream);
#ifdef qmail
  return 100;
#endif
  return 1;
}





char *
strlwr (char *s)
{				// Our own rather crude but working implementation of strlwr() as defined in string.h of some compilers.
  int i;
  for (i = 0; i <= strlen (s); i++)
    s[i] = tolower (s[i]);
  return s;
}
